/*
 *  LICENSE:
 *  Copyright (c) 2008 Freescale Semiconductor
 *  
 *  Permission is hereby granted, free of charge, to any person 
 *  obtaining a copy of this software and associated documentation 
 *  files (the "Software"), to deal in the Software without 
 *  restriction, including without limitation the rights to use, 
 *  copy, modify, merge, publish, distribute, sublicense, and/or 
 *  sell copies of the Software, and to permit persons to whom the 
 *  Software is furnished to do so, subject to the following 
 *  conditions:
 *  
 *  The above copyright notice and this permission notice 
 *  shall be included in all copies or substantial portions 
 *  of the Software.
 *  
 *  THIS SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, 
 *  EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES 
 *  OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND 
 *  NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT 
 *  HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, 
 *  WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, 
 *  OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER 
 *  DEALINGS IN THE SOFTWARE.
 *  
 *
 *  File: $Id: deb_tools.h,v 1.4 2008/08/01 20:57:20 b17778 Exp $
 *
 */



/////////////////////////////////////////////////////////////////////////////////////////////
//                                                                                         //
// This is a samlpe code for the inter process communication between a PPC application and //
// an AXE task running on the AXE scheduler. 
// In this example two FIR filter tasks are loaded to the AXE and then executed in parallel
// with different inputs.
//                                            //
// The following actions are performed:
// 1.	Initialization of the PPC-AXE-driver
// 2.	Initialization and allocation of shared memory objects, like in- and output buffers//
//  	and message memory
// 3.	Initialization and starting of the tasks
// 4.	Display results
// 5.	Free shared memory
// 6.	Unload tasks
//                                                                                         //
/////////////////////////////////////////////////////////////////////////////////////////////



/*
 * include PPC-AXE-driver
 */
#include "os_depend.h"

/*
 * commontypes.h defines the generic common types between AXE and PPC,
 * whereas fir_ipc.h defines special types for the fir-task and the PPC 
 */
#include "commontypes.h"

/*
 * axedatatype-conv.h offers function for type conversion between AXE and PPC
 */
#include "axedatatype-conv.h"

/*
 * filter specific headers
 */
#include "fir.h"
#include "fir_ipc.h" 

/*
 * data.h contains the test data
 */
#include "data.h"

/*
 * standard libraries
 */
#include <stdio.h>

/*
 * open file functionality
 */
#include <fcntl.h>	
#include <sys/stat.h>


/*
 * DEFINES
 */
#define MAX_AXE_MEM_OBJ 40 	// maximal number of shared memory objects

#define MAX_MSG_SIZE 36		// maximal size in bytes for each ipc - message
#define MAX_MSGS 8			// maximal number of messages that can be allocated


/*
 * a struct for a shared memory object
 * pMem: pointer to the memory
 * size: size in bytes
 * region: external or internal memory
 * name: an identifier name
 */
typedef struct 
{
	void* pMem;
	int size;
	int region;
	const char *name;
} sAxeMemObject;


/*
 * GLOBAL VARIABLES
 */

sAxeMemObject* gapAxeMemObjects[MAX_AXE_MEM_OBJ];	// representation of all shared memory objects

MsgRefType gapMsgFreeList[MAX_MSGS];				// representation of used/unused message memory


/**************************************************************************************************
 * the following section contains utilities to handle shared memory. A systematic use of shared 
 * memory is crucial to avoid unpredictable failures. Hence, every shared memory object  should
 * be registerred to gapAxeMemObjects. At the end of the program the function exitAxeMemObjects()
 * MUST be called to ensure that there stays no memory artifacts in the AXE
 */

/**
 *	\brief	 initialize gapAxeMemObjects 
 *	\return    void
 *
 *  \note
 *  
 *  This method should be called at the beginning of each PPC-AXE application to 
 *  enable systematic shared memory controll
 */
void initAxeMemObjects()
{
	int i = 0;
	for(i=0; i<MAX_AXE_MEM_OBJ; i++)
	{
		gapAxeMemObjects[i] = NULL;
	}
	
	return;
}

/**
 *	\brief	 frees all registerred shared memory objects 
 *	\return    void
 *
 *  \note
 *  
 *  This method should be called at the end of each PPC-AXE application to 
 *  free all shared memory
 */
void exitAxeMemObjects()
{
	int i = 0;
	for(i=0; i<MAX_AXE_MEM_OBJ; i++)
	{
		if(gapAxeMemObjects[i] != NULL)
		{
			AxeFree(gapAxeMemObjects[i]->pMem
				   ,gapAxeMemObjects[i]->size
				   ,gapAxeMemObjects[i]->region);
				   
			gapAxeMemObjects[i] = NULL;
		}
	}
	
	return;
}


/**
 *	\brief	 allocates shared memory and register this memory to gapAxeMemObjects
 *  \param[out] the content of that passed pointer will be set to the allocated memory
 *  \param[in] size in bytes to be allocated
 *  \param[in] region where to allocate memory
 *	\return   0 if allocation was succesful
 *
 *  \note
 *  
 */
int addAxeMemObject(void **ptr, unsigned size, int region, const char *name)
{
	int i = 0,
		status = -1;
	for(i = 0; i<MAX_AXE_MEM_OBJ; i++)
	{
		if(gapAxeMemObjects[i] == NULL)
		{
			status = AxeMalloc(ptr, size, region);
			gapAxeMemObjects[i] = (sAxeMemObject*) 
					malloc(sizeof(sAxeMemObject));
			gapAxeMemObjects[i]->pMem = *ptr;
			gapAxeMemObjects[i]->size = size;
			gapAxeMemObjects[i]->region = region;
			gapAxeMemObjects[i]->name = name;
			
			return status;
			
		}
	}
	return status;
}

/**
 *	\brief	 frees the specified shared memory object
 *  \param[in] pointer to the memory, which should be freed
 *	\return   >=0 if freeing was succesful
 *
 *  \note
 *  
 */
int removeAxeMemoObject(void* pMem)
{
	int i = 0;
	for(i = 0; i<MAX_AXE_MEM_OBJ; i++)
	{
		if(gapAxeMemObjects[i]!=NULL 
		   && gapAxeMemObjects[i]->pMem == pMem)
		{
		
			AxeFree(gapAxeMemObjects[i]->pMem
				   ,gapAxeMemObjects[i]->size
				   ,gapAxeMemObjects[i]->region);
				   
			gapAxeMemObjects[i] = NULL;
			return i;
		}
	}
	return -1;
}


/**************************************************************************************************
 * the following section contains utilities to handle ipc messages
 */


/**
 *	\brief	 allocate memory for a message
 *	\return   a pointer to the message memory
 *
 *  \note
 *  
 */

MsgRefType allocMsg (void)
{
    MsgRefType pMsg = NULL;
    int i = 0;

    for (i = 0; i < MAX_MSGS; i++)
    {
        pMsg = gapMsgFreeList[i];
        if (pMsg != NULL)
        {
            gapMsgFreeList[i] = NULL;
            break;
        }
    }
    return pMsg;
}


/**
 *	\brief	 frees memory of message
 *	\param[in]  pointer to the message memory
 *	\return void
 *
 *  \note
 *  
 */
void freeMsg (MsgRefType pMsgIn)
{
    MsgRefType pMsg = NULL;
    int i = 0;

    for (i = 0; i < MAX_MSGS; i++)
    {
        pMsg = gapMsgFreeList[i];
        if (pMsg == NULL)
        {
            gapMsgFreeList[i] = pMsgIn;
            break;
        }
    }
    return;
}

/**
 *	\brief	 this funtion allocates memory for MAX_MSGS messages of MAX_MSG_SIZE size
 *	\param[in]  pointer to the message memory
 *  \return void
 *
 *  \note
 *  This function should be called at the beginning of each ipc application to enable fast
 *  message allocation while running
 */
void initMsgAlloc (void)
{
    int i = 0,
    	status = -1;
    
    for (i = 0; i < MAX_MSGS; i++)
    {
        void *ptr;

        status = addAxeMemObject(&ptr, MAX_MSG_SIZE
				, AXE_MALLOC_EXTERNAL, "msg");
        
        if ((status != AXE_E_OK) || (ptr == NULL))
            printf("failed to allocate memory to message array\n");
        gapMsgFreeList[i] = (MsgRefType)ptr;

    }
}


/**************************************************************************************************
 * the following section contains utilities load binary AXE tasks to the scheduler
 */

/**
 *	\brief	 reads a binary file to a buffer
 *	\param[in]  path and filename of the binary. i.g. /tmp/myBinary.bin
 *  \return pointer to the buffer containing the binary data
 *
 *  \note
 */
void *readBinFile(const char *path)
{
	int fd   = -1,   /* file descriptor */
		size = -1;	 /* file size */
		
	struct stat st;	 /* status */

	void *buf = NULL;/* buffer */

	printf("loading axe binary from %s\n", path);

	/* open file with read only option */
	fd = open(path, O_RDONLY);
	if (fd < 0) {
		return NULL;
	}

	if (fstat(fd, &st)!=0) {
	    close(fd);
	    return NULL;
	}
	if (!S_ISREG(st.st_mode)) {
	    close(fd);
	    return NULL;
	}
	size = st.st_size;

	/* allocate shared memory for the buffer */
	addAxeMemObject((void **)&buf, size, AXE_MALLOC_EXTERNAL, path);
	
	read(fd, buf, size);

	close(fd);

	return buf;
}


/**
 *	\brief	 loads a binary AXE task to the AXE scheduler
 *	\param[in]  path and filename of the binary. i.g. /tmp/myBinary.bin
 *  \param[out]  a pointer to the process ID the AXE scheduler will assign 
 *  \return void
 *
 *  \note
 */
void loadTaskBinToAxe(char *path, int *pid)
{
	int status = -1;			// status variable
	void *taskfilebuf = NULL;	// pointer to task binary code in buffer 

	/* read binary task file to a buffer */
	taskfilebuf = readBinFile(path);
	
	/* load the task to the AXE */
	status = AXELoadTask((const void *)taskfilebuf, 1);
	
	printf("AXELoadTask returned status code %d\n", status);

	if (status < AXE_E_OK) {
		printf("failure in loadTaskBinToAxe");
	}
	
	*pid = status;
}


/**************************************************************************************************
 * the following section contains the main code for this sample.
 * It is avoided to encapsulate some potential functions to augment readability and to maintain 
 * the sequential order.
 */

int main(int argc,  char *argv[])
{

	// VARIABLE DECLARATIONS
	
	int 			taskID1, 	// the tasks ID
					i,			// counter
					status = 0;	// a variable to check system status
		
	__accum  	 	  *pInput1 = NULL,	// Pointer to FIR input
					 *pOutput1 = NULL;	// Pointer to FIR output
		 
	__sFirState   *psFirState1 = NULL;	// The FIR state struct
	__accum 		  *pState1 = NULL;  // Pointer to the FIR state array
	
	__sFirParams *psFirParams1 = NULL;  // The FIR parameter struct
	__fixed 		  *pCoefs1 = NULL;  // Pointer to the coefficients array
	
	sCalcMessage   *psCalcMsg1 = NULL;  // Pointer to a message
	
	double 				 d = 0; // a double used for the print out of the result

		
	char 		*pcTaskBinFile = NULL;	// string that contains path and name of the FIR task binary


	// GET PASSED ARGUMENTS
	if (argc>0) 
	{
		pcTaskBinFile = argv[1];
	}
	else
	{
		printf("not sufficient arguments");
		return 0;
	}


	//INTIALIZATION
	
	/*
	 * initialize the AXE client (ppc-axe-driver)
	 */
	AxeClientInit(); 

	/*
	 * initialize AXE memory object array
	 */
	initAxeMemObjects();
	
	/*
	 * initialize AXE messages
	 */
	initMsgAlloc();

	
	// LOAD AXE TASK BINARIES
	
	loadTaskBinToAxe(pcTaskBinFile, &taskID1);

	/* controlled exit on error */	 
	if(taskID1 < AXE_E_OK) 
	{
		exitAxeMemObjects();
		return 0;
	}
	
	//START AXE TASK
	
	if(status = AXEStartTask(taskID1))
	{
		printf("AXEStartTask returned error number %d\n", status);
		exitAxeMemObjects();
		return 0;
	}


	//ALLOCATE MEMORY FOR FIR FILTER
	
	status += addAxeMemObject(
								(void*) &pInput1
								, sizeof(__accum)*INPUT_SIZE
								, AXE_MALLOC_EXTERNAL
								, "input1"
							  );
							  
	status += addAxeMemObject(
								(void*) &pOutput1
								, sizeof(__accum)*INPUT_SIZE
								, AXE_MALLOC_EXTERNAL
								, "output1"
							  );
							  
	status += addAxeMemObject(
								(void*) &psFirParams1
								, sizeof(__sFirParams)
								, AXE_MALLOC_EXTERNAL
								, "firparams1"
							  );

	status += addAxeMemObject(
								(void*) &pCoefs1
								, sizeof(__fixed)*COEF_SIZE
								, AXE_MALLOC_EXTERNAL
								, "coefs1"
				  			  );
				  			  
	status += addAxeMemObject(
								(void*) &psFirState1
								, sizeof(__sFirState)
								, AXE_MALLOC_EXTERNAL
								, "firstate1"
							  );
							  
	status += addAxeMemObject(
								(void*) &pState1
								, sizeof(__accum)*(COEF_SIZE-1)
								, AXE_MALLOC_EXTERNAL
								, "states1"
							  );
	/* check whether allocation failed */ 						  
	if(status!=0)
	{
		printf("Allocation error! Can't allocate memory\n");
		exitAxeMemObjects();
		return 0;
	}
	
	// INITIALIZE FIR MEMORY
	
	/* initialize input */
	for(i=0; i<INPUT_SIZE; i++)
	{	
		/* convert from the double input array to __accum */
		__doubleToAccum((__double*) (aInput1 + i) , (pInput1 + i));
	}

	/* initialize coefficients */
	for(i=0;i<COEF_SIZE;i++)
	{
		/* convert from the double coefficient array to __fixed */
		__doubleToFixed((__double*) (aCoef + i),(pCoefs1 + i));
	}
	
	/* set output array to 0 */
	memset(pOutput1, 0x00, sizeof(__accum)*INPUT_SIZE);
	
	/* set FIR state array to 0 */
	memset(pState1, 0x00, sizeof(__accum)*(COEF_SIZE-1));
	
	/* set parameters struct */
	psFirParams1->iCoefCount = COEF_SIZE;
	psFirParams1->afiCoef = AxeMapVirtToPhys(pCoefs1); // Pointer has to be mapped to physical memory for AXE use

	/* set state struct */
	psFirState1->iCircBufOffset = 0;
	psFirState1->iFilterStages = COEF_SIZE - 1;
	psFirState1->aacZ = AxeMapVirtToPhys(pState1);	   // Pointer has to be mapped to physical memory for AXE use

	
	// SEND MESSAGE TO AXE TASK
	
	/* allocate message slot */
	psCalcMsg1 = (sCalcMessage*) allocMsg();
	
	/* initialize message */	
	
	/* message header */
	psCalcMsg1->header.recipientID 	= taskID1;					 // task ID
	psCalcMsg1->header.senderID 	= 100;						 // PPC application ID (arbitrary)
	psCalcMsg1->header.type        	= (unsigned int) CALCMSG;	 // The messages type
	psCalcMsg1->header.size 	= sizeof(sCalcMessage);			 // size of the message in bytes
	
	/* message body */
	psCalcMsg1->iBufferCount 	= INPUT_SIZE;					 // number of the entries in the input array
	psCalcMsg1->aacInput 		= AxeMapVirtToPhys(pInput1); 	 // mapped input
	psCalcMsg1->aacOutput 		= AxeMapVirtToPhys(pOutput1);	 // mapped output
	psCalcMsg1->psFirState 		= AxeMapVirtToPhys(psFirState1); // mapped state structure
	psCalcMsg1->psFirParams 	= AxeMapVirtToPhys(psFirParams1);// mapped parameter structure
	
	
	/*
	 * send the message to the task
	 */
	AXESendMessage((void*) psCalcMsg1);
	
	/*
	 * wait for message reply
	 */
	if(AXEReceiveMessageReply(psCalcMsg1, 5000) == 0)
	{
		freeMsg((MsgType*) psCalcMsg1);
	}
	else
	{	
		printf("message 0x%8x (type) from task %d reply failed"
					,psCalcMsg1->header.type
					,psCalcMsg1->header.recipientID );
		exitAxeMemObjects(); // controlled exit
		return 0;
	}


	// PRINT RESULTS
	
	
	for(i=0; i<INPUT_SIZE ; i++)
	{
		/* convert results back to double format */
		__accumToDouble(((__accum*) pOutput1) + i, (__double*) &d);
		
		/* print result: index: result */
		printf("%4d:\t",  i);
		printf("%2.5f\n",  d  );		
	}


	// CLEAN UP
	
	/* 
	 * unload task 
	 */
	if(AXEUnloadTask(taskID1))	printf("error unloading task 1");

	/* 
	 * free all shared memory objects
	 */
	exitAxeMemObjects();


	// EXIT
	return 0;
}
